<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    redirect(APP_URL . '/admin/login.php');
}

$pageTitle = 'Analytics';
$adminId = $_SESSION['admin_id'];

// Date range
$startDate = isset($_GET['start_date']) ? sanitize($_GET['start_date']) : date('Y-m-d', strtotime('-30 days'));
$endDate = isset($_GET['end_date']) ? sanitize($_GET['end_date']) : date('Y-m-d');

// Get page views by date
$pageViewsQuery = "
    SELECT DATE(visit_date) as date, COUNT(*) as count 
    FROM page_views 
    WHERE visit_date BETWEEN ? AND ? 
    GROUP BY DATE(visit_date) 
    ORDER BY date
";
$pageViews = $db->select($pageViewsQuery, [$startDate, $endDate . ' 23:59:59']);

// Get link clicks by date
$linkClicksQuery = "
    SELECT DATE(visit_date) as date, COUNT(*) as count 
    FROM link_clicks 
    WHERE visit_date BETWEEN ? AND ? 
    GROUP BY DATE(visit_date) 
    ORDER BY date
";
$linkClicks = $db->select($linkClicksQuery, [$startDate, $endDate . ' 23:59:59']);

// Get top 10 books by views
$topBooksQuery = "
    SELECT b.id, b.title, u.username, COUNT(pv.id) as views
    FROM books b
    JOIN users u ON b.user_id = u.id
    JOIN page_views pv ON pv.page_id = b.id AND pv.page_type = 'book'
    WHERE pv.visit_date BETWEEN ? AND ?
    GROUP BY b.id
    ORDER BY views DESC
    LIMIT 10
";
$topBooks = $db->select($topBooksQuery, [$startDate, $endDate . ' 23:59:59']);

// Get top 10 links by clicks
$topLinksQuery = "
    SELECT bt.id, bt.label, bt.url, b.title, COUNT(lc.id) as clicks
    FROM buttons bt
    JOIN books b ON bt.book_id = b.id
    JOIN link_clicks lc ON lc.button_id = bt.id
    WHERE lc.visit_date BETWEEN ? AND ?
    GROUP BY bt.id
    ORDER BY clicks DESC
    LIMIT 10
";
$topLinks = $db->select($topLinksQuery, [$startDate, $endDate . ' 23:59:59']);

// Get referrer statistics
$referrersQuery = "
    SELECT referrer, COUNT(*) as count
    FROM page_views
    WHERE visit_date BETWEEN ? AND ?
    GROUP BY referrer
    ORDER BY count DESC
    LIMIT 10
";
$referrers = $db->select($referrersQuery, [$startDate, $endDate . ' 23:59:59']);

// Include header
include 'templates/header.php';
?>

<div class="d-flex justify-content-between align-items-center mb-4">
    <h1 class="h3">Analytics</h1>
    
    <form class="d-flex" action="">
        <div class="input-group me-2">
            <span class="input-group-text">From</span>
            <input type="date" name="start_date" class="form-control" value="<?php echo $startDate; ?>">
        </div>
        <div class="input-group me-2">
            <span class="input-group-text">To</span>
            <input type="date" name="end_date" class="form-control" value="<?php echo $endDate; ?>">
        </div>
        <button type="submit" class="btn btn-primary">Apply</button>
    </form>
</div>

<!-- Engagement Overview -->
<div class="row">
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Page Views</h6>
            </div>
            <div class="card-body">
                <canvas id="pageViewsChart" width="100%" height="300"></canvas>
            </div>
        </div>
    </div>
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Link Clicks</h6>
            </div>
            <div class="card-body">
                <canvas id="linkClicksChart" width="100%" height="300"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- Top Content -->
<div class="row">
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Top Links by Clicks</h6>
            </div>
            <div class="card-body">
                <?php if (count($topLinks) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Link</th>
                                    <th>Book</th>
                                    <th>Clicks</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($topLinks as $link): ?>
                                    <tr>
                                        <td>
                                            <a href="<?php echo sanitize($link['url']); ?>" target="_blank">
                                                <?php echo sanitize($link['label']); ?>
                                            </a>
                                        </td>
                                        <td><?php echo sanitize($link['title']); ?></td>
                                        <td><?php echo $link['clicks']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No data available for the selected date range</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<!-- Referrers -->
<div class="row">
    <div class="col-lg-12">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Top Referrers</h6>
            </div>
            <div class="card-body">
                <?php if (count($referrers) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Referrer</th>
                                    <th>Count</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($referrers as $referrer): ?>
                                    <tr>
                                        <td><?php echo empty($referrer['referrer']) ? 'Direct / None' : sanitize($referrer['referrer']); ?></td>
                                        <td><?php echo $referrer['count']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No data available for the selected date range</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
document.addEventListener('DOMContentLoaded', function() {
    // Prepare data for the page views chart
    const pageViewsData = {
        labels: [
            <?php
            $dates = [];
            $views = [];
            
            // Create date range array
            $current = strtotime($startDate);
            $end = strtotime($endDate);
            
            while ($current <= $end) {
                $currentDate = date('Y-m-d', $current);
                $dates[] = "'" . date('M j', $current) . "'";
                
                // Find views for this date
                $found = false;
                foreach ($pageViews as $view) {
                    if ($view['date'] == $currentDate) {
                        $views[] = $view['count'];
                        $found = true;
                        break;
                    }
                }
                
                if (!$found) {
                    $views[] = 0;
                }
                
                $current = strtotime('+1 day', $current);
            }
            
            echo implode(', ', $dates);
            ?>
        ],
        datasets: [{
            label: 'Page Views',
            data: [<?php echo implode(', ', $views); ?>],
            backgroundColor: 'rgba(78, 115, 223, 0.05)',
            borderColor: 'rgba(78, 115, 223, 1)',
            pointRadius: 3,
            pointBackgroundColor: 'rgba(78, 115, 223, 1)',
            pointBorderColor: 'rgba(78, 115, 223, 1)',
            pointHoverRadius: 3,
            pointHoverBackgroundColor: 'rgba(78, 115, 223, 1)',
            pointHoverBorderColor: 'rgba(78, 115, 223, 1)',
            pointHitRadius: 10,
            pointBorderWidth: 2,
            lineTension: 0.3
        }]
    };
    
    // Prepare data for the link clicks chart
    const linkClicksData = {
        labels: [
            <?php
            $clicks = [];
            
            // Reuse dates from page views
            // Create date range array
            $current = strtotime($startDate);
            $end = strtotime($endDate);
            
            while ($current <= $end) {
                $currentDate = date('Y-m-d', $current);
                
                // Find clicks for this date
                $found = false;
                foreach ($linkClicks as $click) {
                    if ($click['date'] == $currentDate) {
                        $clicks[] = $click['count'];
                        $found = true;
                        break;
                    }
                }
                
                if (!$found) {
                    $clicks[] = 0;
                }
                
                $current = strtotime('+1 day', $current);
            }
            
            // Dates are already echoed in page views chart
            ?>
        ],
        datasets: [{
            label: 'Link Clicks',
            data: [<?php echo implode(', ', $clicks); ?>],
            backgroundColor: 'rgba(28, 200, 138, 0.05)',
            borderColor: 'rgba(28, 200, 138, 1)',
            pointRadius: 3,
            pointBackgroundColor: 'rgba(28, 200, 138, 1)',
            pointBorderColor: 'rgba(28, 200, 138, 1)',
            pointHoverRadius: 3,
            pointHoverBackgroundColor: 'rgba(28, 200, 138, 1)',
            pointHoverBorderColor: 'rgba(28, 200, 138, 1)',
            pointHitRadius: 10,
            pointBorderWidth: 2,
            lineTension: 0.3
        }]
    };
    
    // Draw page views chart
    const pageViewsChart = new Chart(document.getElementById('pageViewsChart'), {
        type: 'line',
        data: pageViewsData,
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        precision: 0
                    }
                }
            }
        }
    });
    
    // Draw link clicks chart
    const linkClicksChart = new Chart(document.getElementById('linkClicksChart'), {
        type: 'line',
        data: linkClicksData,
        options: {
            responsive: true,
            maintainAspectRatio: false,
            scales: {
                y: {
                    beginAtZero: true,
                    ticks: {
                        precision: 0
                    }
                }
            }
        }
    });
});
</script>

<?php include 'templates/footer.php'; ?>