<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Check if admin is logged in
if (!isset($_SESSION['admin_id'])) {
    redirect(APP_URL . '/admin/login.php');
}

$pageTitle = 'Dashboard';
$adminId = $_SESSION['admin_id'];

// Get some statistics
$totalUsers = $db->selectOne("SELECT COUNT(*) as count FROM users")['count'];
$totalBooks = $db->selectOne("SELECT COUNT(*) as count FROM books")['count'];
$totalLinks = $db->selectOne("SELECT COUNT(*) as count FROM buttons")['count'];

// Get recent page views (last 7 days)
$recentViews = $db->select("
    SELECT DATE(visit_date) as date, COUNT(*) as count 
    FROM page_views 
    WHERE visit_date >= DATE_SUB(NOW(), INTERVAL 7 DAY) 
    GROUP BY DATE(visit_date) 
    ORDER BY date
");

// Get recent user registrations (last 7 days)
$recentUsers = $db->select("
    SELECT DATE(created_at) as date, COUNT(*) as count 
    FROM users 
    WHERE created_at >= DATE_SUB(NOW(), INTERVAL 7 DAY) 
    GROUP BY DATE(created_at) 
    ORDER BY date
");

// Get most viewed books
$mostViewedBooks = $db->select("
    SELECT b.id, b.title, u.username, COUNT(pv.id) as views
    FROM books b
    JOIN users u ON b.user_id = u.id
    JOIN page_views pv ON pv.page_id = b.id AND pv.page_type = 'book'
    GROUP BY b.id
    ORDER BY views DESC
    LIMIT 5
");

// Get most clicked links
$mostClickedLinks = $db->select("
    SELECT bt.label, bt.url, b.title, COUNT(lc.id) as clicks
    FROM buttons bt
    JOIN books b ON bt.book_id = b.id
    JOIN link_clicks lc ON lc.button_id = bt.id
    GROUP BY bt.id
    ORDER BY clicks DESC
    LIMIT 5
");

// Include header
include 'templates/header.php';
?>

<h1 class="h3 mb-4">Dashboard</h1>

<!-- Statistics Cards -->
<div class="row">
    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card stat-card-primary h-100">
            <div class="card-body">
                <div class="row no-gutters align-items-center">
                    <div class="col mr-2">
                        <div class="text-xs font-weight-bold text-primary text-uppercase mb-1">Total Users</div>
                        <div class="h5 mb-0 font-weight-bold"><?php echo $totalUsers; ?></div>
                    </div>
                    <div class="col-auto">
                        <i class="fas fa-users fa-2x text-gray-300 stat-icon"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card stat-card-success h-100">
            <div class="card-body">
                <div class="row no-gutters align-items-center">
                    <div class="col mr-2">
                        <div class="text-xs font-weight-bold text-success text-uppercase mb-1">Total Books</div>
                        <div class="h5 mb-0 font-weight-bold"><?php echo $totalBooks; ?></div>
                    </div>
                    <div class="col-auto">
                        <i class="fas fa-book fa-2x text-gray-300 stat-icon"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card stat-card-info h-100">
            <div class="card-body">
                <div class="row no-gutters align-items-center">
                    <div class="col mr-2">
                        <div class="text-xs font-weight-bold text-info text-uppercase mb-1">Total Links</div>
                        <div class="h5 mb-0 font-weight-bold"><?php echo $totalLinks; ?></div>
                    </div>
                    <div class="col-auto">
                        <i class="fas fa-link fa-2x text-gray-300 stat-icon"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>

    <div class="col-xl-3 col-md-6 mb-4">
        <div class="card stat-card stat-card-warning h-100">
            <div class="card-body">
                <div class="row no-gutters align-items-center">
                    <div class="col mr-2">
                        <div class="text-xs font-weight-bold text-warning text-uppercase mb-1">Avg. Links Per Book</div>
                        <div class="h5 mb-0 font-weight-bold"><?php echo $totalBooks > 0 ? round($totalLinks / $totalBooks, 1) : 0; ?></div>
                    </div>
                    <div class="col-auto">
                        <i class="fas fa-chart-line fa-2x text-gray-300 stat-icon"></i>
                    </div>
                </div>
            </div>
        </div>
    </div>
</div>

<!-- Charts -->
<div class="row">
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Page Views (Last 7 Days)</h6>
            </div>
            <div class="card-body">
                <canvas id="pageViewsChart" width="100%" height="40"></canvas>
            </div>
        </div>
    </div>
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">New Registrations (Last 7 Days)</h6>
            </div>
            <div class="card-body">
                <canvas id="registrationsChart" width="100%" height="40"></canvas>
            </div>
        </div>
    </div>
</div>

<!-- Content Row -->
<div class="row">
    <!-- Most Viewed Books -->
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Most Viewed Books</h6>
            </div>
            <div class="card-body">
                <?php if (count($mostViewedBooks) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Title</th>
                                    <th>Author</th>
                                    <th>Views</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($mostViewedBooks as $book): ?>
                                    <tr>
                                        <td><?php echo sanitize($book['title']); ?></td>
                                        <td><?php echo sanitize($book['username']); ?></td>
                                        <td><?php echo $book['views']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No data available yet</p>
                <?php endif; ?>
            </div>
        </div>
    </div>

    <!-- Most Clicked Links -->
    <div class="col-lg-6">
        <div class="card mb-4">
            <div class="card-header">
                <h6 class="m-0 font-weight-bold">Most Clicked Links</h6>
            </div>
            <div class="card-body">
                <?php if (count($mostClickedLinks) > 0): ?>
                    <div class="table-responsive">
                        <table class="table table-bordered table-hover" width="100%" cellspacing="0">
                            <thead>
                                <tr>
                                    <th>Label</th>
                                    <th>Book</th>
                                    <th>Clicks</th>
                                </tr>
                            </thead>
                            <tbody>
                                <?php foreach ($mostClickedLinks as $link): ?>
                                    <tr>
                                        <td><?php echo sanitize($link['label']); ?></td>
                                        <td><?php echo sanitize($link['title']); ?></td>
                                        <td><?php echo $link['clicks']; ?></td>
                                    </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                <?php else: ?>
                    <p class="text-center text-muted">No data available yet</p>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<script>
// Chart initialization
document.addEventListener('DOMContentLoaded', function() {
    // Page Views Chart
    const pageViewsCtx = document.getElementById('pageViewsChart');
    if (pageViewsCtx) {
        const pageViewsChart = new Chart(pageViewsCtx, {
            type: 'line',
            data: {
                labels: [
                    <?php 
                    $dates = [];
                    $views = [];
                    
                    // Get last 7 days
                    for ($i = 6; $i >= 0; $i--) {
                        $date = date('Y-m-d', strtotime("-$i days"));
                        $dates[] = "'".date('M j', strtotime($date))."'";
                        
                        // Find views for this date
                        $found = false;
                        foreach ($recentViews as $view) {
                            if ($view['date'] == $date) {
                                $views[] = $view['count'];
                                $found = true;
                                break;
                            }
                        }
                        
                        if (!$found) {
                            $views[] = 0;
                        }
                    }
                    
                    echo implode(', ', $dates);
                    ?>
                ],
                datasets: [{
                    label: 'Page Views',
                    data: [<?php echo implode(', ', $views); ?>],
                    lineTension: 0.3,
                    backgroundColor: 'rgba(78, 115, 223, 0.05)',
                    borderColor: 'rgba(78, 115, 223, 1)',
                    pointRadius: 3,
                    pointBackgroundColor: 'rgba(78, 115, 223, 1)',
                    pointBorderColor: 'rgba(78, 115, 223, 1)',
                    pointHoverRadius: 5,
                    pointHoverBackgroundColor: 'rgba(78, 115, 223, 1)',
                    pointHoverBorderColor: 'rgba(78, 115, 223, 1)',
                    pointHitRadius: 10,
                    pointBorderWidth: 2
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }
    
    // Registrations Chart
    const registrationsCtx = document.getElementById('registrationsChart');
    if (registrationsCtx) {
        const registrationsChart = new Chart(registrationsCtx, {
            type: 'bar',
            data: {
                labels: [
                    <?php 
                    $dates = [];
                    $registrations = [];
                    
                    // Get last 7 days
                    for ($i = 6; $i >= 0; $i--) {
                        $date = date('Y-m-d', strtotime("-$i days"));
                        $dates[] = "'".date('M j', strtotime($date))."'";
                        
                        // Find registrations for this date
                        $found = false;
                        foreach ($recentUsers as $user) {
                            if ($user['date'] == $date) {
                                $registrations[] = $user['count'];
                                $found = true;
                                break;
                            }
                        }
                        
                        if (!$found) {
                            $registrations[] = 0;
                        }
                    }
                    
                    echo implode(', ', $dates);
                    ?>
                ],
                datasets: [{
                    label: 'New Users',
                    data: [<?php echo implode(', ', $registrations); ?>],
                   backgroundColor: 'rgba(28, 200, 138, 0.8)',
                    borderColor: 'rgba(28, 200, 138, 1)',
                    borderWidth: 1
                }]
            },
            options: {
                responsive: true,
                maintainAspectRatio: false,
                scales: {
                    y: {
                        beginAtZero: true,
                        ticks: {
                            precision: 0
                        }
                    }
                }
            }
        });
    }
});
</script>

<?php include 'templates/footer.php'; ?>