<?php
require_once '../includes/config.php';
require_once '../includes/db.php';
require_once '../includes/functions.php';

// Redirect if not logged in
if (!isLoggedIn()) {
    setFlashMessage('danger', 'Please log in to access this page');
    redirect(APP_URL . '/pages/login.php');
}

$pageTitle = 'Add/Edit Book';
$userId = $_SESSION['user_id'];
$errors = [];
$book = [
    'id' => null,
    'title' => '',
    'description' => '',
    'cover_image' => '',
    'background_color' => '#ffffff',
    'text_color' => '#000000',
    'button_color' => '#0066ff',
    'button_text_color' => '#ffffff',
    'is_published' => 1
];

// Check if editing existing book
if (isset($_GET['id'])) {
    $bookId = (int)$_GET['id'];
    $existingBook = getBookById($bookId);
    
    // Verify book exists and belongs to current user
    if ($existingBook && $existingBook['user_id'] == $userId) {
        $book = $existingBook;
    } else {
        setFlashMessage('danger', 'Book not found or you do not have permission to edit it');
        redirect(APP_URL . '/pages/dashboard.php');
    }
}

// Handle form submission
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $book['title'] = sanitize($_POST['title']);
    $book['description'] = sanitize($_POST['description']);
    $book['background_color'] = sanitize($_POST['background_color']);
    $book['text_color'] = sanitize($_POST['text_color']);
    $book['button_color'] = sanitize($_POST['button_color']);
    $book['button_text_color'] = sanitize($_POST['button_text_color']);
    $book['is_published'] = isset($_POST['is_published']) ? 1 : 0;
    
    // Validate input
    if (empty($book['title'])) {
        $errors[] = 'Book title is required';
    }
    
    // Handle cover image upload
    if (!empty($_FILES['cover_image']['name'])) {
        $uploadDir = UPLOAD_DIR;
        
        // Create upload directory if it doesn't exist
        if (!file_exists($uploadDir)) {
            mkdir($uploadDir, 0755, true);
        }
        
        $coverImage = uploadImage($_FILES['cover_image'], $uploadDir);
        
        if ($coverImage) {
            $book['cover_image'] = $coverImage;
        } else {
            $errors[] = 'Invalid image file. Please upload a JPG, PNG, or GIF file.';
        }
    }
    
    // If no errors, save book
    if (empty($errors)) {
        if ($book['id']) {
            // Update existing book
            $bookData = [
                'title' => $book['title'],
                'description' => $book['description'],
                'background_color' => $book['background_color'],
                'text_color' => $book['text_color'],
                'button_color' => $book['button_color'],
                'button_text_color' => $book['button_text_color'],
                'is_published' => $book['is_published']
            ];
            
            if (!empty($book['cover_image'])) {
                $bookData['cover_image'] = $book['cover_image'];
            }
            
            $db->update('books', $bookData, 'id = ?', [$book['id']]);
            setFlashMessage('success', 'Book updated successfully');
        } else {
            // Create new book
            $bookData = [
                'user_id' => $userId,
                'title' => $book['title'],
                'description' => $book['description'],
                'background_color' => $book['background_color'],
                'text_color' => $book['text_color'],
                'button_color' => $book['button_color'],
                'button_text_color' => $book['button_text_color'],
                'is_published' => $book['is_published'],
                'cover_image' => $book['cover_image'] ?? null
            ];
            
            $bookId = $db->insert('books', $bookData);
            setFlashMessage('success', 'Book created successfully');
        }
        
        redirect(APP_URL . '/pages/dashboard.php');
    }
}

include '../templates/header.php';
?>

<div class="row">
    <div class="col-md-8">
        <div class="card">
            <div class="card-header">
                <h3><?php echo $book['id'] ? 'Edit Book' : 'Add New Book'; ?></h3>
            </div>
            <div class="card-body">
                <?php if (!empty($errors)): ?>
                    <div class="alert alert-danger">
                        <ul class="mb-0">
                            <?php foreach ($errors as $error): ?>
                                <li><?php echo $error; ?></li>
                            <?php endforeach; ?>
                        </ul>
                    </div>
                <?php endif; ?>
                
                <form method="POST" enctype="multipart/form-data">
                    <div class="mb-3">
                        <label for="title" class="form-label">Book Title *</label>
                        <input type="text" id="title" name="title" class="form-control" value="<?php echo $book['title']; ?>" required>
                    </div>
                    
                    <div class="mb-3">
                        <label for="description" class="form-label">Description</label>
                        <textarea id="description" name="description" class="form-control" rows="5"><?php echo $book['description']; ?></textarea>
                        <small class="form-text text-muted">Provide a brief description of your book</small>
                    </div>
                    
                    <div class="mb-3">
                        <label for="cover_image" class="form-label">Cover Image</label>
                        <?php if (!empty($book['cover_image'])): ?>
                            <div class="mb-2">
                                <img src="<?php echo UPLOAD_URL . $book['cover_image']; ?>" alt="Current cover" class="img-thumbnail" style="max-height: 200px;">
                            </div>
                        <?php endif; ?>
                        <input type="file" id="cover_image" name="cover_image" class="form-control">
                        <small class="form-text text-muted">Upload a JPG, PNG, or GIF image for your book cover</small>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="background_color" class="form-label">Background Color</label>
                            <input type="color" id="background_color" name="background_color" class="form-control form-control-color" value="<?php echo $book['background_color']; ?>">
                        </div>
                        <div class="col-md-6">
                            <label for="text_color" class="form-label">Text Color</label>
                            <input type="color" id="text_color" name="text_color" class="form-control form-control-color" value="<?php echo $book['text_color']; ?>">
                        </div>
                    </div>
                    
                    <div class="row mb-3">
                        <div class="col-md-6">
                            <label for="button_color" class="form-label">Button Color</label>
                            <input type="color" id="button_color" name="button_color" class="form-control form-control-color" value="<?php echo $book['button_color']; ?>">
                        </div>
                        <div class="col-md-6">
                            <label for="button_text_color" class="form-label">Button Text Color</label>
                            <input type="color" id="button_text_color" name="button_text_color" class="form-control form-control-color" value="<?php echo $book['button_text_color']; ?>">
                        </div>
                    </div>
                    
                    <div class="mb-3 form-check">
                        <input type="checkbox" id="is_published" name="is_published" class="form-check-input" <?php echo $book['is_published'] ? 'checked' : ''; ?>>
                        <label for="is_published" class="form-check-label">Publish this book</label>
                    </div>
                    
                    <div class="d-flex justify-content-between">
                        <a href="<?php echo APP_URL; ?>/pages/dashboard.php" class="btn btn-secondary">Cancel</a>
                        <button type="submit" class="btn btn-primary">Save Book</button>
                    </div>
                </form>
            </div>
        </div>
    </div>
    
    <div class="col-md-4">
        <div class="card">
            <div class="card-header">
                <h3>Tips</h3>
            </div>
            <div class="card-body">
                <ul>
                    <li>Choose colors that match your book's theme or your author brand.</li>
                    <li>Make sure there's enough contrast between text and background colors for readability.</li>
                    <li>Upload a high-quality cover image for the best appearance.</li>
                    <li>Keep your description concise and engaging.</li>
                </ul>
                
                <?php if ($book['id']): ?>
                    <div class="mt-4">
                        <a href="<?php echo APP_URL; ?>/pages/edit-buttons.php?book_id=<?php echo $book['id']; ?>" class="btn btn-info w-100 mb-2">Manage Book Links</a>
                        <a href="<?php echo APP_URL . '/' . getUserById($userId)['permalink']; ?>" target="_blank" class="btn btn-success w-100">Preview Book Page</a>
                    </div>
                <?php endif; ?>
            </div>
        </div>
    </div>
</div>

<?php include '../templates/footer.php'; ?>